#!/usr/bin/perl
#
# Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved. 
#
################ Documentation ################

# The SYNOPSIS section is printed out as usage when incorrect parameters
# are passed

=head1 NAME

  roothas.pl - root configuration script for SI-HAS

=head1 SYNOPSIS

  roothas.pl [-verbose] [-upgrade | -patch]
             [-paramfile <parameter-file>] [-deconfig] [-force]
             [-unlock [-hahome <path to Oracle Restart home> [-nocrsstop]]
             [[-prepatch | -postpatch] [-norestart]] [-lockacfs]
             [-init]

  Options:
   -verbose   Run this script in verbose mode
   -upgrade   Oracle HA is being upgraded from previous version
   -patch     Oracle HA is being upgraded to a patch version
   -paramfile Complete path of file specifying HA parameter values
   -deconfig  To deconfigure Oracle Restart
   -force     To deconfigure Oracle Restart when configuration
              is in a good condition
   -unlock    Unlock Oracle Restart home
   -hahome    Complete path of Oracle HA home
   -nocrsstop Used with unlock option to reset permissions on an inactive Oracle HA home
   -prepatch  Perform required steps before the Oracle patching tool (Opatch) is invoked
   -postpatch Perform required steps after the Oracle patching tool (Opatch) is invoked
   -norestart Leave SIHA stack down after patching
   -lockacfs  Lock ACFS files for use as super user
   -init      Reset the permissions of all files and directories under Oracle HA home

  If neither -upgrade nor -patch is supplied, a new install is performed

  To see the full manpage for this program, execute:
    perldoc roothas.pl

=head1 DESCRIPTION

  This script performs the operations necessary to install the Oracle
  SingleInstance-HighAvailability stack.  This is run once during the
  configuration of SI-HAS home.

  This script does the following:
  1) Setup permissions of binaries and DSOs in SI-HAS home
  2) Setup OLR for storing SI-HAS configuration data

  Upon successful completion on each node, the Oracle SI-HA stack will
  be executing.

=cut

################ End Documentation ################

#    MODIFIED   (MM/DD/YY)
#    xyuan       08/31/12 - Fix bug 14535011 - Add '-init' option
#    sidshank    04/20/12 - adding -auto option to be used internally by root
#                           automation alone.
#    xyuan       02/22/12 - Add -prepatch & -postpatch
#    anjiwaji    11/30/11 - Add -lockacfs function.
#    xyuan       10/05/11 - Fix bug 13054957
#    sidshank    08/11/11 - removing check_Superuser call related to bug
#                           12739826
#    dpham       04/10/11 - New for 12c
#
use strict;
use English;
use File::Spec::Functions;
use File::Basename;
use Getopt::Long qw(:config no_auto_abbrev);
use Pod::Usage;

BEGIN {
   # Add the directory of this file to the search path
   push @INC, dirname($PROGRAM_NAME);
}

# root scripts modules
use crsinstall;
use crsupgrade;
use crsdeconfig;
use crspatch;
use crsutils;

my ($DEBUG, $DECONFIG, $FORCE, $UPGRADE, $INIT,
    $PATCH, $UNLOCK, $NOCRSSTOP, $LOCK_ACFS, $HELP,$AUTO) = 0;
my $hahome = "";

my ($PREPATCH, $POSTPATCH, $NORESTART) = 0;

# pull all parameters defined in crsconfig_params (if it exists)
# as variables in Perl
my $PARAM_FILE = catfile (dirname ($0), "crsconfig_params");

# pull all parameters defined in crsconfig_params and s_crsconfig_defs (if
# it exists) as variables in Perl
my $osdfile = catfile (dirname ($0), "s_crsconfig_defs");

# Parse command line args
# If an incorrect option is specified, the perl POD at the beginning
# of the file is parsed into a man page
# the return code to give when the incorrect parameters are passed
my $usage_rc = 1;

GetOptions('verbose!'    => \$DEBUG,
           'upgrade!',   => \$UPGRADE,
           'deconfig!',  => \$DECONFIG,
           'force'       => \$FORCE,
           'patch!'      => \$PATCH,
           'unlock'      => \$UNLOCK,
           'hahome=s'    => \$hahome,
           'paramfile=s' => \$PARAM_FILE,
           'nocrsstop!'  => \$NOCRSSTOP,
           'prepatch'    => \$PREPATCH,
           'postpatch'   => \$POSTPATCH,
           'norestart!'  => \$NORESTART,
           'lockacfs'    => \$LOCK_ACFS,
           'auto'        => \$AUTO,
           'init'        => \$INIT,
	   'help!'       => \$HELP
          ) or pod2usage($usage_rc);

# Check validity of args
pod2usage(-msg => "Invalid extra options passed: @ARGV",
          -exitval => $usage_rc) if (@ARGV);

#-------------------------------------------------------------------------------
# MAIN SCRIPT BODY
#-------------------------------------------------------------------------------
local $SIG{'__DIE__'} = sub { dietrap(@_); };
local $SIG{INT}       = sub { dietrap(@_); };

if (! $DEBUG) { $DEBUG = $ENV{'ORA_INSTALL_DEBUG'}; }

if ($HELP) {
	   pod2usage(0);
}

elsif ($DECONFIG) {
   crsdeconfig->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    CRSDeconfig         => $DECONFIG,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    FORCE               => $FORCE,
                    crscfg_trace        => TRUE,
		    AUTO                => $AUTO
                   );
}
elsif ($UPGRADE) {
    crsupgrade->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    UPGRADE             => $UPGRADE,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    crscfg_trace        => TRUE,
		    AUTO                => $AUTO
                   );
}
elsif ($PATCH) {
      crspatch->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    HAPatch             => $PATCH,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    crscfg_trace        => TRUE
                   );
}
elsif ($UNLOCK) { 
      crsutils->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    UNLOCK              => $UNLOCK,
                    hahome              => $hahome,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    NOCRSSTOP           => $NOCRSSTOP,
                    crscfg_trace        => TRUE
                   );


   my $exclfile = catfile($CFG->ORA_CRS_HOME, 'crs', 'install', 'install.excl');
   unlockHAHomefordeinstall($exclfile);
}
elsif ($PREPATCH) {
      crspatch->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    PREPATCH            => $PREPATCH,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    NORESTART           => $NORESTART,
                    crscfg_trace        => TRUE
                   );
}
elsif ($POSTPATCH) {
      crspatch->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    POSTPATCH           => $POSTPATCH,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    NORESTART           => $NORESTART,
                    crscfg_trace        => TRUE
                   );
}
elsif ($LOCK_ACFS) { 
      crsutils->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    LOCK_ACFS           => $LOCK_ACFS,
                    hahome              => $hahome,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    crscfg_trace        => TRUE
                   );
   lockAcfsFiles();
}
elsif ($INIT) {
    crsinstall->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    crscfg_trace        => TRUE,
                    init                => $INIT
                   );
}
else { # fresh install
    crsinstall->new(SIHA                => TRUE,
                    DEBUG               => $DEBUG,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    crscfg_trace        => TRUE,
		    AUTO                => $AUTO
                   );
}

0;
